using System;
using System.Collections;
using Method = System.Reflection.MethodBase;
using System.Drawing.Printing;
using System.Data;
using System.Text;
using System.Text.RegularExpressions;
using SECTION = gov.va.med.vbecs.Common.DatabaseConstants.AdminReportSection;
using SUBDIVISION = gov.va.med.vbecs.Common.DatabaseConstants.AdminReportSubdivision;
using COLUMN = gov.va.med.vbecs.Common.DatabaseConstants.AdminReportColumnNames;
using Microsoft.Reporting.WinForms;
using System.Reflection;
using System.IO;
using TABLE = gov.va.med.vbecs.Common.VbecsTables;
using ARTIFICIAL = gov.va.med.vbecs.Common.DatabaseConstants.ArtificialColumnNames;

namespace gov.va.med.vbecs.BOL
{
	#region Header

	///<Package>Package: VBECS - VistA Blood Establishment Computer System</Package>
	///<Warning> WARNING: Per VHA Directive $VADIRECTIVE this class should not be modified</Warning>
	///<MedicalDevice> Medical Device #: $MEDDEVICENO</MedicalDevice>
	///<Developers>
	///<Developer>Margaret Jablonski</Developer>
	///</Developers>
	///<SiteName>Hines OIFO</SiteName>
	///<CreationDate>8/24/2004</CreationDate>
	///<Note>The Food and Drug Administration classifies this software as a medical device.  As such, it may not be changed in any way. Modifications to this software may result in an adulterated medical device under 21CFR820, the use of which is considered to be a violation of US Federal Statutes.  Acquiring and implementing this software through the Freedom of information Act requires the implementor to assume total responsibility for the software, and become a registered manufacturer of a medical device, subject to FDA regulations</Note>
	///<summary>Summary description for AdministrativeDataReport.</summary>

	#endregion

	public class AdministrativeDataReport:VbecsReport
	{
		#region Variables

		private DataTable dtReport = null; 
		private DataRow[] dtExisting = null;
		private DataTable dtDiscarded;
		private DataTable dtModified;
		private DataTable dtOutdated;
		private DataTable dtReceived;
		private DataTable dtShipped;
		private DataTable dtTransfused;

		#endregion
		
		#region Consructors

		
		///<Developers>
		///	<Developer>Margaret Jablonski</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>10/20/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="5825"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>Initialized AdministrativeData object</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="5826"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///<summary>Blank constructor for AdministrativeDataReport.</summary>
		public AdministrativeDataReport()
		{
		}
		#endregion

		#region Methods

		///<Developers>
		///	<Developer>Margaret Jablonski</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>10/20/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="5823"> 
		///		<ExpectedInput>Valid Transfusion Requiremets Report object</ExpectedInput>
		///		<ExpectedOutput>DataTable containing Transfusion Requirements report data for a single division</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="5824"> 
		///		<ExpectedInput>Invalid division code</ExpectedInput>
		///		<ExpectedOutput>Empty DataTable</ExpectedOutput>
		///	</Case>
		///
		///<Case type="0" testid ="6166"> 
		///		<ExpectedInput>Valid division code, multidivisional</ExpectedInput>
		///		<ExpectedOutput>DataTable containing Transfusion Requirements report data for multiple divisions</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Generate data source for Administrative Data Report
		/// </summary>
		/// <returns></returns>
		public DataTable GetAdministrativeDataReport()
		{
			dtDiscarded = DAL.BloodUnit.GetAdminReportBloodUnitsDiscarded(this.StartDate, this.EndDate, this.IsMultiDivisional, this.DivisionCode);
			dtModified = DAL.BloodUnit.GetAdminReportBloodUnitsModifiedPrepared(this.StartDate, this.EndDate, this.IsMultiDivisional, this.DivisionCode);
			dtOutdated = DAL.BloodUnit.GetAdminReportBloodUnitsOutdated(this.StartDate, this.IsMultiDivisional, this.DivisionCode);
			dtReceived = DAL.BloodUnit.GetAdminReportBloodUnitsReceived(this.StartDate, this.EndDate, this.IsMultiDivisional, this.DivisionCode);
			dtShipped = DAL.BloodUnit.GetAdminReportBloodUnitsShipped(this.StartDate, this.EndDate, this.IsMultiDivisional,  this.DivisionCode);
			dtTransfused = DAL.BloodUnit.GetAdminReportBloodUnitsTransfused(this.StartDate, this.EndDate, this.IsMultiDivisional, this.DivisionCode);

			int recordCount = dtDiscarded.Rows.Count +
							dtModified.Rows.Count +
							dtOutdated.Rows.Count +
							dtReceived.Rows.Count +
							dtShipped.Rows.Count +
							dtTransfused.Rows.Count;

			if (recordCount == 0)
			{
				return new DataTable();
			}
			else 
			{
				MergeAdminReportData();
				
				DataSet _ds = new DataSet();
				_ds.Tables.Add(dtReport);
				return _ds.Tables[0];
			}
		}

		/// <summary>
		/// MergeAdminReportData
		/// </summary>
		private void MergeAdminReportData()					
		{

			dtReport = this.GetAdministrativeDataReportSchema();

			if(dtDiscarded.Rows.Count > 0)
			{
				AddEachCategory("Discarded", dtDiscarded);
			}
			if(dtModified.Rows.Count > 0)
			{
				AddModifiedCategory(dtModified);
			}
			if(dtOutdated.Rows.Count > 0)
			{
				AddEachCategory("Outdated", dtOutdated);
			}
			if(dtReceived.Rows.Count > 0)
			{
				AddEachCategory("Received", dtReceived);
			}
			if(dtShipped.Rows.Count > 0)
			{
				AddEachCategory("Shipped", dtShipped);
			}
			if(dtTransfused.Rows.Count > 0)
			{
				AddEachCategory(ARTIFICIAL.Transfused, dtTransfused);
			}
		}

		/// <summary>
		/// AddEachCategory
		/// </summary>
		/// <param name="categoryName"></param>
		/// <param name="dtCategoryData"></param>
		private void AddEachCategory(string categoryName, DataTable dtCategoryData)
		{
			int donationTypeID;
			int componentClassID;
			string allogeneicColumn, directedColumn, autologousColumn;

			foreach(DataRow dr in dtCategoryData.Rows)
			{
				//Check if we are dealing with multi-division report
				CheckDivision(dr);

				//Get Donation Type ID
				donationTypeID = (int) dr[TABLE.BloodUnit.DonationTypeId];
	
				//Get Component Class ID
				componentClassID = (int) dr[TABLE.ComponentClass.ComponentClassId];

				//Get Product Type Code
				Common.ProductType productType = Common.Utility.GetProductTypeFromProductTypeCode(dr[TABLE.ProductType.ProductTypeCode].ToString());

				allogeneicColumn = string.Empty;
				directedColumn = string.Empty;
				autologousColumn = string.Empty;

				switch(categoryName)
				{
					case "Discarded":
						allogeneicColumn = COLUMN.DiscardedAllogeneic;
						directedColumn = COLUMN.DiscardedDirected;
						autologousColumn = COLUMN.DiscardedAutologous;
						break;
					case "Modified":
						allogeneicColumn = COLUMN.ModifiedAllogeneic;
						directedColumn = COLUMN.ModifiedDirected;
						autologousColumn = COLUMN.ModifiedAutologous;
						break;
					case "Outdated":
						allogeneicColumn = COLUMN.OutdatedAllogeneic;
						directedColumn = COLUMN.OutdatedDirected;
						autologousColumn = COLUMN.OutdatedAutologous;
						break;
					case "Received":
						allogeneicColumn = COLUMN.ReceivedAllogeneic;
						directedColumn = COLUMN.ReceivedDirected;
						autologousColumn = COLUMN.ReceivedAutologous;
						break;
					case "Shipped":
						allogeneicColumn = COLUMN.ShippedAllogeneic;
						directedColumn = COLUMN.ShippedDirected;
						autologousColumn = COLUMN.ShippedAutologous;
						break;
					case ARTIFICIAL.Transfused:
						allogeneicColumn = COLUMN.TransfusedAllogeneic;
						directedColumn = COLUMN.TransfusedDirected;
						autologousColumn = COLUMN.TransfusedAutologous;
						break;
				}

				switch(componentClassID)
				{
					case (int) Common.ComponentClass.RBC:
						
						UpdateTotals(0, donationTypeID, allogeneicColumn, directedColumn, autologousColumn);

						//Split/Divided
						if(IsISBTSplit(dr) || IsCodabarSplit(dr))
						{
							UpdateTotals(1, donationTypeID, allogeneicColumn, directedColumn, autologousColumn);
						}

						//Irradiate Cells
						if(dr[TABLE.BloodProduct.ProductAttributeForm].ToString().ToUpper().IndexOf("C2") !=  -1)
						{
							UpdateTotals(2, donationTypeID, allogeneicColumn, directedColumn, autologousColumn);
						}

						//Leukoreduced
						if(dr[TABLE.BloodProduct.ProductAttributeForm].ToString().ToUpper().IndexOf("D1") !=  -1 ||
							dr[TABLE.BloodProduct.ProductAttributeForm].ToString().ToUpper().IndexOf("D2") !=  -1 ||
							dr[TABLE.BloodProduct.ProductAttributeForm].ToString().ToUpper().IndexOf("D3") !=  -1 ||
							dr[TABLE.BloodProduct.ProductAttributeForm].ToString().ToUpper().IndexOf("D4") !=  -1 ||
							dr[TABLE.BloodProduct.ProductAttributeForm].ToString().ToUpper().IndexOf("D5") !=  -1 ||
							dr[TABLE.BloodProduct.ProductAttributeForm].ToString().ToUpper().IndexOf("D6") !=  -1 ||
							dr[TABLE.BloodProduct.ProductAttributeForm].ToString().ToUpper().IndexOf("D7") !=  -1 ||
							dr[TABLE.BloodProduct.ProductAttributeForm].ToString().ToUpper().IndexOf("D8") !=  -1 ||
							dr[TABLE.BloodProduct.ProductAttributeForm].ToString().ToUpper().IndexOf("D9") !=  -1)
						{
							UpdateTotals(3, donationTypeID, allogeneicColumn, directedColumn, autologousColumn);
						}

						//Wash RBC
						if(productType == Common.ProductType.WashedRedBloodCells ||
							(productType == Common.ProductType.WashedApheresisRedBloodCells && categoryName == "Received"))
						{
							UpdateTotals(4, donationTypeID, allogeneicColumn, directedColumn, autologousColumn);
						}
					                
						//Deglycerolized
						if(productType == Common.ProductType.DeglycerolizedRedBloodCells ||
							productType == Common.ProductType.DeglycerolizedRejuvenatedRedBloodCells)
						{
							UpdateTotals(5, donationTypeID, allogeneicColumn, directedColumn, autologousColumn);
						}

						// CMV Negative
						// CR 2105: need to make sure the value is set before actually counting it
						if(!dr.IsNull(Common.DatabaseConstants.ArtificialColumnNames.CMVNegative)
							&& (int)dr[Common.DatabaseConstants.ArtificialColumnNames.CMVNegative] > 0)
						{
							UpdateTotals(6, donationTypeID, allogeneicColumn, directedColumn, autologousColumn);
						}

						// SC Negative
						// CR 2105: need to make sure the value is set before actually counting it
						if(!dr.IsNull(Common.DatabaseConstants.ArtificialColumnNames.SCNegative)
							&& (int)dr[Common.DatabaseConstants.ArtificialColumnNames.SCNegative] > 0)
						{
							UpdateTotals(7, donationTypeID, allogeneicColumn, directedColumn, autologousColumn);
						}

						//Frozen
						if(productType == Common.ProductType.FrozenRedBloodCells)
						{
							UpdateTotals(8, donationTypeID, allogeneicColumn, directedColumn, autologousColumn);
						}

						//Rejuvenated
						if(productType == Common.ProductType.RejuvenatedRedBloodCells)
						{
							UpdateTotals(9, donationTypeID, allogeneicColumn, directedColumn, autologousColumn);
						}

						//Frozen/Rejuvenated
						if(productType == Common.ProductType.FrozenRejuvenatedRedBloodCells)
						{
							UpdateTotals(10, donationTypeID, allogeneicColumn, directedColumn, autologousColumn);
						}
						break;

						//volume reduce for Modified only

					case (int) Common.ComponentClass.WB:

						UpdateTotals(12, donationTypeID, allogeneicColumn, directedColumn, autologousColumn);

						//Split/Divided
						if(IsISBTSplit(dr) || IsCodabarSplit(dr))
						{
							UpdateTotals(13, donationTypeID, allogeneicColumn, directedColumn, autologousColumn);
						}

						//Irradiate Cells
						if(dr[TABLE.BloodProduct.ProductAttributeForm].ToString().ToUpper().IndexOf("C2") !=  -1)
						{
							UpdateTotals(14, donationTypeID, allogeneicColumn, directedColumn, autologousColumn);
						}
						
						//Wash RBC
						if(productType == Common.ProductType.WashedRedBloodCells ||
							(productType == Common.ProductType.WholeBlood && categoryName == "Received"))
						{
							UpdateTotals(15, donationTypeID, allogeneicColumn, directedColumn, autologousColumn);
						}

						//Leukoreduced
						if(dr[TABLE.BloodProduct.ProductAttributeForm].ToString().ToUpper().IndexOf("D1") !=  -1 ||
							dr[TABLE.BloodProduct.ProductAttributeForm].ToString().ToUpper().IndexOf("D2") !=  -1 ||
							dr[TABLE.BloodProduct.ProductAttributeForm].ToString().ToUpper().IndexOf("D3") !=  -1 ||
							dr[TABLE.BloodProduct.ProductAttributeForm].ToString().ToUpper().IndexOf("D4") !=  -1 ||
							dr[TABLE.BloodProduct.ProductAttributeForm].ToString().ToUpper().IndexOf("D5") !=  -1 ||
							dr[TABLE.BloodProduct.ProductAttributeForm].ToString().ToUpper().IndexOf("D6") !=  -1 ||
							dr[TABLE.BloodProduct.ProductAttributeForm].ToString().ToUpper().IndexOf("D7") !=  -1 ||
							dr[TABLE.BloodProduct.ProductAttributeForm].ToString().ToUpper().IndexOf("D8") !=  -1 ||
							dr[TABLE.BloodProduct.ProductAttributeForm].ToString().ToUpper().IndexOf("D9") !=  -1)
						{
							UpdateTotals(16, donationTypeID, allogeneicColumn, directedColumn, autologousColumn);
						}		             

						// CMV Negative
						// CR 2105: need to make sure the value is set before actually counting it
						if(!dr.IsNull(Common.DatabaseConstants.ArtificialColumnNames.CMVNegative)
							&& (int)dr[Common.DatabaseConstants.ArtificialColumnNames.CMVNegative] > 0)
						{
							UpdateTotals(17, donationTypeID, allogeneicColumn, directedColumn, autologousColumn);
						}

						// SC Negative
						// CR 2105: need to make sure the value is set before actually counting it
						if(!dr.IsNull(Common.DatabaseConstants.ArtificialColumnNames.SCNegative)
							&& (int)dr[Common.DatabaseConstants.ArtificialColumnNames.SCNegative] > 0)
						{
							UpdateTotals(18, donationTypeID, allogeneicColumn, directedColumn, autologousColumn);
						}

						//Volume reduce
						if(dr[TABLE.BloodUnit.DiscardedPlasmaVolume] != DBNull.Value && 
							Convert.ToInt32(dr[TABLE.BloodUnit.DiscardedPlasmaVolume]) > 0 && 
							categoryName != "Received")
						{
							UpdateTotals(21, donationTypeID, allogeneicColumn, directedColumn, autologousColumn);
						}
						break;

					case (int) Common.ComponentClass.FFP:

						UpdateTotals(22, donationTypeID, allogeneicColumn, directedColumn, autologousColumn);

						//Thawed (FFP)
						if(productType == Common.ProductType.ThawedFreshFrozenPlasma ||
							productType == Common.ProductType.ThawedApheresisFreshFrozenPlasma)
						{
							UpdateTotals(23, donationTypeID, allogeneicColumn, directedColumn, autologousColumn);
						}

						//Leukoreduced
						if(dr[TABLE.BloodProduct.ProductAttributeForm].ToString().ToUpper().IndexOf("D1") !=  -1 ||
							dr[TABLE.BloodProduct.ProductAttributeForm].ToString().ToUpper().IndexOf("D2") !=  -1 ||
							dr[TABLE.BloodProduct.ProductAttributeForm].ToString().ToUpper().IndexOf("D3") !=  -1 ||
							dr[TABLE.BloodProduct.ProductAttributeForm].ToString().ToUpper().IndexOf("D4") !=  -1 ||
							dr[TABLE.BloodProduct.ProductAttributeForm].ToString().ToUpper().IndexOf("D5") !=  -1 ||
							dr[TABLE.BloodProduct.ProductAttributeForm].ToString().ToUpper().IndexOf("D6") !=  -1 ||
							dr[TABLE.BloodProduct.ProductAttributeForm].ToString().ToUpper().IndexOf("D7") !=  -1 ||
							dr[TABLE.BloodProduct.ProductAttributeForm].ToString().ToUpper().IndexOf("D8") !=  -1 ||
							dr[TABLE.BloodProduct.ProductAttributeForm].ToString().ToUpper().IndexOf("D9") !=  -1)
						{
							UpdateTotals(24, donationTypeID, allogeneicColumn, directedColumn, autologousColumn);
						}

						//Split/Divided
						if((IsISBTSplit(dr) || IsCodabarSplit(dr)) &&
							(productType == Common.ProductType.ThawedFreshFrozenPlasma ||
							productType == Common.ProductType.ThawedApheresisFreshFrozenPlasma))
						{
							UpdateTotals(25, donationTypeID, allogeneicColumn, directedColumn, autologousColumn);
						}

						//Irradiated 
						if((productType == Common.ProductType.ThawedFreshFrozenPlasma ||
							productType == Common.ProductType.ThawedApheresisFreshFrozenPlasma) &&
							dr[TABLE.BloodProduct.ProductAttributeForm].ToString().ToUpper().IndexOf("C2") !=  -1)

						{
							UpdateTotals(26, donationTypeID, allogeneicColumn, directedColumn, autologousColumn);
						}
						break;

					case (int) Common.ComponentClass.PLT:

						UpdateTotals(27, donationTypeID, allogeneicColumn, directedColumn, autologousColumn);

						//Pooled
						if(productType == Common.ProductType.PooledPlatelets ||
							productType == Common.ProductType.WashedPooledPlatelets)
						{
							UpdateTotals(28, donationTypeID, allogeneicColumn, directedColumn, autologousColumn);
						}

						//Irradiate Cells
						if(dr[TABLE.BloodProduct.ProductAttributeForm].ToString().ToUpper().IndexOf("C2") !=  -1)
						{
							UpdateTotals(29, donationTypeID, allogeneicColumn, directedColumn, autologousColumn);
						}

						//Wash Platelets 
						if(productType == Common.ProductType.WashedPlatelets ||
							productType == Common.ProductType.WashedPooledPlatelets)
						{
							UpdateTotals(30, donationTypeID, allogeneicColumn, directedColumn, autologousColumn);
						}

						//Leukoreduced
						if(dr[TABLE.BloodProduct.ProductAttributeForm].ToString().ToUpper().IndexOf("D1") !=  -1 ||
							dr[TABLE.BloodProduct.ProductAttributeForm].ToString().ToUpper().IndexOf("D2") !=  -1 ||
							dr[TABLE.BloodProduct.ProductAttributeForm].ToString().ToUpper().IndexOf("D3") !=  -1 ||
							dr[TABLE.BloodProduct.ProductAttributeForm].ToString().ToUpper().IndexOf("D4") !=  -1 ||
							dr[TABLE.BloodProduct.ProductAttributeForm].ToString().ToUpper().IndexOf("D5") !=  -1 ||
							dr[TABLE.BloodProduct.ProductAttributeForm].ToString().ToUpper().IndexOf("D6") !=  -1 ||
							dr[TABLE.BloodProduct.ProductAttributeForm].ToString().ToUpper().IndexOf("D7") !=  -1 ||
							dr[TABLE.BloodProduct.ProductAttributeForm].ToString().ToUpper().IndexOf("D8") !=  -1 ||
							dr[TABLE.BloodProduct.ProductAttributeForm].ToString().ToUpper().IndexOf("D9") !=  -1)
						{
							UpdateTotals(31, donationTypeID, allogeneicColumn, directedColumn, autologousColumn);
						}

						// CMV Negative
						// CR 2105: need to make sure the value is set before actually counting it
						if(!dr.IsNull(Common.DatabaseConstants.ArtificialColumnNames.CMVNegative)
							&& (int)dr[Common.DatabaseConstants.ArtificialColumnNames.CMVNegative] > 0)
						{
							UpdateTotals(32, donationTypeID, allogeneicColumn, directedColumn, autologousColumn);
						}
						break;

					case (int) Common.ComponentClass.CRYO:

						UpdateTotals(33, donationTypeID, allogeneicColumn, directedColumn, autologousColumn);

						//Pooled
						if(productType == Common.ProductType.PooledCryoprecipitate)
						{
							UpdateTotals(34, donationTypeID, allogeneicColumn, directedColumn, autologousColumn);
						}

						//Thawed
						if(productType == Common.ProductType.ThawedCryoprecipitate ||
							productType == Common.ProductType.ThawedApheresisCryoprecipitate)
						{
							UpdateTotals(35, donationTypeID, allogeneicColumn, directedColumn, autologousColumn);
						}

						//Thawed/Pooled
						if(productType == Common.ProductType.ThawedPooledCryoprecipitate)
						{
							UpdateTotals(36, donationTypeID, allogeneicColumn, directedColumn, autologousColumn);
						}

						break;

					case (int) Common.ComponentClass.Other:

						UpdateTotals(37, donationTypeID, allogeneicColumn, directedColumn, autologousColumn);


						//Split/Divided
						if(IsISBTSplit(dr) || IsCodabarSplit(dr))
						{
							UpdateTotals(38, donationTypeID, allogeneicColumn, directedColumn, autologousColumn);
						}

						//Irradiate Cells
						if(dr[TABLE.BloodProduct.ProductAttributeForm].ToString().ToUpper().IndexOf("C2") !=  -1)
						{
							UpdateTotals(39, donationTypeID, allogeneicColumn, directedColumn, autologousColumn);
						}

						//Leukoreduced
						if(dr[TABLE.BloodProduct.ProductAttributeForm].ToString().ToUpper().IndexOf("D1") !=  -1 ||
							dr[TABLE.BloodProduct.ProductAttributeForm].ToString().ToUpper().IndexOf("D2") !=  -1 ||
							dr[TABLE.BloodProduct.ProductAttributeForm].ToString().ToUpper().IndexOf("D3") !=  -1 ||
							dr[TABLE.BloodProduct.ProductAttributeForm].ToString().ToUpper().IndexOf("D4") !=  -1 ||
							dr[TABLE.BloodProduct.ProductAttributeForm].ToString().ToUpper().IndexOf("D5") !=  -1 ||
							dr[TABLE.BloodProduct.ProductAttributeForm].ToString().ToUpper().IndexOf("D6") !=  -1 ||
							dr[TABLE.BloodProduct.ProductAttributeForm].ToString().ToUpper().IndexOf("D7") !=  -1 ||
							dr[TABLE.BloodProduct.ProductAttributeForm].ToString().ToUpper().IndexOf("D8") !=  -1 ||
							dr[TABLE.BloodProduct.ProductAttributeForm].ToString().ToUpper().IndexOf("D9") !=  -1)
						{
							UpdateTotals(40, donationTypeID, allogeneicColumn, directedColumn, autologousColumn);
						}

						// CMV Negative
						// CR 2105: need to make sure the value is set before actually counting it
						if(!dr.IsNull(Common.DatabaseConstants.ArtificialColumnNames.CMVNegative)
							&& (int)dr[Common.DatabaseConstants.ArtificialColumnNames.CMVNegative] > 0)
						{
							UpdateTotals(41, donationTypeID, allogeneicColumn, directedColumn, autologousColumn);
						}

						//Pooled 
						if(productType == Common.ProductType.PooledGranulocytes ||
							productType == Common.ProductType.PooledPlasma ||
							productType == Common.ProductType.PooledPlateletRichBuffyCoat)
						{
							UpdateTotals(42, donationTypeID, allogeneicColumn, directedColumn, autologousColumn);
						}
						break;
				}
			}
		}

		/// <summary>
		/// AddModifiedCategory
		/// </summary>
		/// <param name="dtCategoryData"></param>
		private void AddModifiedCategory(DataTable dtCategoryData)
		{
			//Since this category is different in TT, it's coded separately from 'Add Each Category'
			int donationTypeID;
			int componentClassID;
			string allogeneicColumn, directedColumn, autologousColumn;

			foreach(DataRow dr in dtCategoryData.Rows)
			{
				//Check if we are dealing with multi-division report
				CheckDivision(dr);

				//Get Donation Type ID
				donationTypeID = (int) dr[TABLE.BloodUnit.DonationTypeId];
	
				//Get Component Class ID
				componentClassID = (int) dr[TABLE.ComponentClass.ComponentClassId];

				//Get Product Type Code
				Common.ProductType productType = Common.Utility.GetProductTypeFromProductTypeCode(dr[TABLE.ProductType.ProductTypeCode].ToString());

				allogeneicColumn = COLUMN.ModifiedAllogeneic;
				directedColumn = COLUMN.ModifiedDirected;
				autologousColumn = COLUMN.ModifiedAutologous;

				switch(componentClassID)
				{
					case (int) Common.ComponentClass.RBC:
						
						UpdateTotals(0, donationTypeID, allogeneicColumn, directedColumn, autologousColumn);

						//Split/Divided
						if(dr[TABLE.BloodUnitModification.ProductModificationCode].ToString().ToUpper() == Common.Utility.GetProductModificationCharFromEnum(Common.ProductModification.SplitAdult).ToString())
						{
							UpdateTotals(1, donationTypeID, allogeneicColumn, directedColumn, autologousColumn);
						}

						//Irradiate Cells
						if(dr[TABLE.BloodUnitModification.ProductModificationCode].ToString().ToUpper() == Common.Utility.GetProductModificationCharFromEnum(Common.ProductModification.Irradiated).ToString())
						{
							UpdateTotals(2, donationTypeID, allogeneicColumn, directedColumn, autologousColumn);
						}

						//Leukoreduced
						if(dr[TABLE.BloodUnitModification.ProductModificationCode].ToString().ToUpper() == Common.Utility.GetProductModificationCharFromEnum(Common.ProductModification.LeukoReduced).ToString())
						{
							UpdateTotals(3, donationTypeID, allogeneicColumn, directedColumn, autologousColumn);
						}

						//Wash RBC
						if(dr[TABLE.BloodUnitModification.ProductModificationCode].ToString().ToUpper() == Common.Utility.GetProductModificationCharFromEnum(Common.ProductModification.Washed).ToString())
						{
							UpdateTotals(4, donationTypeID, allogeneicColumn, directedColumn, autologousColumn);
						}
					                
						//Deglycerolized
						if(dr[TABLE.BloodUnitModification.ProductModificationCode].ToString().ToUpper() == Common.Utility.GetProductModificationCharFromEnum(Common.ProductModification.Deglycerolized).ToString())
						{
							UpdateTotals(5, donationTypeID, allogeneicColumn, directedColumn, autologousColumn);
						}

						//Frozen
						if(dr[TABLE.BloodUnitModification.ProductModificationCode].ToString().ToUpper() == Common.Utility.GetProductModificationCharFromEnum(Common.ProductModification.Frozen).ToString())
						{
							UpdateTotals(8, donationTypeID, allogeneicColumn, directedColumn, autologousColumn);
						}

						//Rejuvenated
						if(dr[TABLE.BloodUnitModification.ProductModificationCode].ToString().ToUpper() == Common.Utility.GetProductModificationCharFromEnum(Common.ProductModification.Rejuvenated).ToString())
						{
							UpdateTotals(9, donationTypeID, allogeneicColumn, directedColumn, autologousColumn);
						}

						//Volume Reduce 
						if(dr[TABLE.BloodUnitModification.ProductModificationCode].ToString().ToUpper() == Common.Utility.GetProductModificationCharFromEnum(Common.ProductModification.VolumeReduced).ToString())
						{
							UpdateTotals(11, donationTypeID, allogeneicColumn, directedColumn, autologousColumn);
						}
						break;

					case (int) Common.ComponentClass.WB:

						UpdateTotals(12, donationTypeID, allogeneicColumn, directedColumn, autologousColumn);

						//Split/Divided
						if(dr[TABLE.BloodUnitModification.ProductModificationCode].ToString().ToUpper() == Common.Utility.GetProductModificationCharFromEnum(Common.ProductModification.SplitAdult).ToString())
						{
							UpdateTotals(13, donationTypeID, allogeneicColumn, directedColumn, autologousColumn);
						}

						//Irradiate Cells
						if(dr[TABLE.BloodUnitModification.ProductModificationCode].ToString().ToUpper() == Common.Utility.GetProductModificationCharFromEnum(Common.ProductModification.Irradiated).ToString())
						{
							UpdateTotals(14, donationTypeID, allogeneicColumn, directedColumn, autologousColumn);
						}
						
						//Wash RBC
						if(dr[TABLE.BloodUnitModification.ProductModificationCode].ToString().ToUpper() == Common.Utility.GetProductModificationCharFromEnum(Common.ProductModification.Washed).ToString())
						{
							UpdateTotals(15, donationTypeID, allogeneicColumn, directedColumn, autologousColumn);
						}

						//Leukoreduced
						if(dr[TABLE.BloodUnitModification.ProductModificationCode].ToString().ToUpper() == Common.Utility.GetProductModificationCharFromEnum(Common.ProductModification.LeukoReduced).ToString())
						{
							UpdateTotals(16, donationTypeID, allogeneicColumn, directedColumn, autologousColumn);
						}
					                
						//Frozen
						if(dr[TABLE.BloodUnitModification.ProductModificationCode].ToString().ToUpper() == Common.Utility.GetProductModificationCharFromEnum(Common.ProductModification.Frozen).ToString())
						{
							UpdateTotals(19, donationTypeID, allogeneicColumn, directedColumn, autologousColumn);
						}

						//Rejuvenated
						if(dr[TABLE.BloodUnitModification.ProductModificationCode].ToString().ToUpper() == Common.Utility.GetProductModificationCharFromEnum(Common.ProductModification.Rejuvenated).ToString())
						{
							UpdateTotals(20, donationTypeID, allogeneicColumn, directedColumn, autologousColumn);
						}

						//Volume Reduce 
						if(dr[TABLE.BloodUnitModification.ProductModificationCode].ToString().ToUpper() == Common.Utility.GetProductModificationCharFromEnum(Common.ProductModification.VolumeReduced).ToString())
						{
							UpdateTotals(21, donationTypeID, allogeneicColumn, directedColumn, autologousColumn);
						}
						break;

					case (int) Common.ComponentClass.FFP:

						UpdateTotals(22, donationTypeID, allogeneicColumn, directedColumn, autologousColumn);

						//Thawed (FFP)
						if(dr[TABLE.BloodUnitModification.ProductModificationCode].ToString().ToUpper() == Common.Utility.GetProductModificationCharFromEnum(Common.ProductModification.Thawed).ToString())
						{
							UpdateTotals(23, donationTypeID, allogeneicColumn, directedColumn, autologousColumn);
						}

						//Leukoreduced
						if(dr[TABLE.BloodUnitModification.ProductModificationCode].ToString().ToUpper() == Common.Utility.GetProductModificationCharFromEnum(Common.ProductModification.LeukoReduced).ToString())
						{
							UpdateTotals(24, donationTypeID, allogeneicColumn, directedColumn, autologousColumn);
						}

						//Split/Divided
						if(dr[TABLE.BloodUnitModification.ProductModificationCode].ToString().ToUpper() == Common.Utility.GetProductModificationCharFromEnum(Common.ProductModification.SplitAdult).ToString())
						{
							UpdateTotals(25, donationTypeID, allogeneicColumn, directedColumn, autologousColumn);
						}

						//Irradiated 
						if(dr[TABLE.BloodUnitModification.ProductModificationCode].ToString().ToUpper() == Common.Utility.GetProductModificationCharFromEnum(Common.ProductModification.Irradiated).ToString())

						{
							UpdateTotals(26, donationTypeID, allogeneicColumn, directedColumn, autologousColumn);
						}
						break;

					case (int) Common.ComponentClass.PLT:

						UpdateTotals(27, donationTypeID, allogeneicColumn, directedColumn, autologousColumn);

						//Pooled
						if(dr[TABLE.BloodUnitModification.ProductModificationCode].ToString().ToUpper() == Common.Utility.GetProductModificationCharFromEnum(Common.ProductModification.Pooled).ToString())
						{
							UpdateTotals(28, donationTypeID, allogeneicColumn, directedColumn, autologousColumn);
							
							//Also update the PooledUnitsCount
							if(IsAllogeneic(donationTypeID))
							{
								dtExisting[28][COLUMN.ModifiedAllogeneicPooledUnits] = (int) dtExisting[28][COLUMN.ModifiedAllogeneicPooledUnits] + Convert.ToInt32(dr[ARTIFICIAL.PooledUnitsCount]);  
							}
							else if(Common.Utility.IsAutologous(donationTypeID))
							{
								dtExisting[28][COLUMN.ModifiedAutologousPooledUnits] = (int) dtExisting[28][COLUMN.ModifiedAutologousPooledUnits] + Convert.ToInt32(dr[ARTIFICIAL.PooledUnitsCount]);  
							}
							else if(Common.Utility.IsDirected(donationTypeID))
							{
								dtExisting[28][COLUMN.ModifiedDirectedPooledUnits] = (int) dtExisting[28][COLUMN.ModifiedDirectedPooledUnits] + Convert.ToInt32(dr[ARTIFICIAL.PooledUnitsCount]);  
							}
						}

						//Irradiate Cells
						if(dr[TABLE.BloodUnitModification.ProductModificationCode].ToString().ToUpper() == Common.Utility.GetProductModificationCharFromEnum(Common.ProductModification.Irradiated).ToString())
						{
							UpdateTotals(29, donationTypeID, allogeneicColumn, directedColumn, autologousColumn);
						}

						//Wash Platelets 
						if(dr[TABLE.BloodUnitModification.ProductModificationCode].ToString().ToUpper() == Common.Utility.GetProductModificationCharFromEnum(Common.ProductModification.Washed).ToString())
						{
							UpdateTotals(30, donationTypeID, allogeneicColumn, directedColumn, autologousColumn);
						}

						//Leukoreduced
						if(dr[TABLE.BloodUnitModification.ProductModificationCode].ToString().ToUpper() == Common.Utility.GetProductModificationCharFromEnum(Common.ProductModification.LeukoReduced).ToString())
						{
							UpdateTotals(31, donationTypeID, allogeneicColumn, directedColumn, autologousColumn);
						}
						break;

					case (int) Common.ComponentClass.CRYO:

						UpdateTotals(33, donationTypeID, allogeneicColumn, directedColumn, autologousColumn);

						//Pooled
						if(dr[TABLE.BloodUnitModification.ProductModificationCode].ToString().ToUpper() == Common.Utility.GetProductModificationCharFromEnum(Common.ProductModification.Pooled).ToString())
						{
							UpdateTotals(34, donationTypeID, allogeneicColumn, directedColumn, autologousColumn);
							
							//Also update the PooledUnitsCount
							if(IsAllogeneic(donationTypeID))
							{
								dtExisting[34][COLUMN.ModifiedAllogeneicPooledUnits] = (int) dtExisting[34][COLUMN.ModifiedAllogeneicPooledUnits] + Convert.ToInt32(dr[ARTIFICIAL.PooledUnitsCount]);  
							}
							else if(Common.Utility.IsAutologous(donationTypeID))
							{
								dtExisting[34][COLUMN.ModifiedAutologousPooledUnits] = (int) dtExisting[34][COLUMN.ModifiedAutologousPooledUnits] + Convert.ToInt32(dr[ARTIFICIAL.PooledUnitsCount]);  
							}
							else if(Common.Utility.IsDirected(donationTypeID))
							{
								dtExisting[34][COLUMN.ModifiedDirectedPooledUnits] = (int) dtExisting[34][COLUMN.ModifiedDirectedPooledUnits] + Convert.ToInt32(dr[ARTIFICIAL.PooledUnitsCount]);  
							}
						}

						//Thawed
						if(dr[TABLE.BloodUnitModification.ProductModificationCode].ToString().ToUpper() == Common.Utility.GetProductModificationCharFromEnum(Common.ProductModification.Thawed).ToString() &&
							(productType == Common.ProductType.ThawedCryoprecipitate ||
							productType == Common.ProductType.ThawedApheresisCryoprecipitate ))
						{
							UpdateTotals(35, donationTypeID, allogeneicColumn, directedColumn, autologousColumn);
						}

						//Thawed/Pooled
						if(dr[TABLE.BloodUnitModification.ProductModificationCode].ToString().ToUpper() == Common.Utility.GetProductModificationCharFromEnum(Common.ProductModification.PoolThawCryo).ToString()&& 
							productType == Common.ProductType.ThawedPooledCryoprecipitate)
						{
							UpdateTotals(36, donationTypeID, allogeneicColumn, directedColumn, autologousColumn);
							
							//Also update the PooledUnitsCount
							if(IsAllogeneic(donationTypeID))
							{
								dtExisting[36][COLUMN.ModifiedAllogeneicPooledUnits] = (int) dtExisting[36][COLUMN.ModifiedAllogeneicPooledUnits] + Convert.ToInt32(dr[ARTIFICIAL.PooledUnitsCount]);  
							}
							else if(Common.Utility.IsAutologous(donationTypeID))
							{
								dtExisting[36][COLUMN.ModifiedAutologousPooledUnits] = (int) dtExisting[36][COLUMN.ModifiedAutologousPooledUnits] + Convert.ToInt32(dr[ARTIFICIAL.PooledUnitsCount]);  
							}
							else if(Common.Utility.IsDirected(donationTypeID))
							{
								dtExisting[36][COLUMN.ModifiedDirectedPooledUnits] = (int) dtExisting[36][COLUMN.ModifiedDirectedPooledUnits] + Convert.ToInt32(dr[ARTIFICIAL.PooledUnitsCount]);  
							}
						}

						break;

					case (int) Common.ComponentClass.Other:

						UpdateTotals(37, donationTypeID, allogeneicColumn, directedColumn, autologousColumn);


						//Split/Divided
						if(dr[TABLE.BloodUnitModification.ProductModificationCode].ToString().ToUpper() == Common.Utility.GetProductModificationCharFromEnum(Common.ProductModification.SplitAdult).ToString())
						{
							UpdateTotals(38, donationTypeID, allogeneicColumn, directedColumn, autologousColumn);
						}

						//Irradiate Cells
						if(dr[TABLE.BloodUnitModification.ProductModificationCode].ToString().ToUpper() == Common.Utility.GetProductModificationCharFromEnum(Common.ProductModification.Irradiated).ToString())
						{
							UpdateTotals(39, donationTypeID, allogeneicColumn, directedColumn, autologousColumn);
						}

						//Leukoreduced
						if(dr[TABLE.BloodUnitModification.ProductModificationCode].ToString().ToUpper() == Common.Utility.GetProductModificationCharFromEnum(Common.ProductModification.LeukoReduced).ToString())
						{
							UpdateTotals(40, donationTypeID, allogeneicColumn, directedColumn, autologousColumn);
						}

						//Pooled 
						if( dr[TABLE.BloodUnitModification.ProductModificationCode].ToString().ToUpper() == Common.Utility.GetProductModificationCharFromEnum(Common.ProductModification.Pooled).ToString() &&
							(productType == Common.ProductType.PooledGranulocytes ||
							productType == Common.ProductType.PooledPlasma ||
							productType == Common.ProductType.PooledPlateletRichBuffyCoat))
						{
							UpdateTotals(42, donationTypeID, allogeneicColumn, directedColumn, autologousColumn);
						
							//Also update the PooledUnitsCount
							if(IsAllogeneic(donationTypeID))
							{
								dtExisting[42][COLUMN.ModifiedAllogeneicPooledUnits] = (int) dtExisting[42][COLUMN.ModifiedAllogeneicPooledUnits] + Convert.ToInt32(dr[ARTIFICIAL.PooledUnitsCount]);  
							}
							else if(Common.Utility.IsAutologous(donationTypeID))
							{
								dtExisting[42][COLUMN.ModifiedAutologousPooledUnits] = (int) dtExisting[42][COLUMN.ModifiedAutologousPooledUnits] + Convert.ToInt32(dr[ARTIFICIAL.PooledUnitsCount]);  
							}
							else if(Common.Utility.IsDirected(donationTypeID))
							{
								dtExisting[42][COLUMN.ModifiedDirectedPooledUnits] = (int) dtExisting[42][COLUMN.ModifiedDirectedPooledUnits] + Convert.ToInt32(dr[ARTIFICIAL.PooledUnitsCount]);  
							}	
						}
						break;
				}
			}
		}

		/// <summary>
		/// GetAdministrativeDataReportSchema
		/// </summary>
		/// <returns></returns>
		private DataTable GetAdministrativeDataReportSchema()
		{
			DataTable dt = new DataTable("AdministrativeDataReport");
			dt.Columns.Add(COLUMN.DivisionCode);
			dt.Columns.Add(COLUMN.ComponentClassName);
			dt.Columns.Add(COLUMN.SubdivisionName);
			dt.Columns.Add(COLUMN.ReceivedAllogeneic, typeof(int));
			dt.Columns.Add(COLUMN.ReceivedDirected, typeof(int)); 
			dt.Columns.Add(COLUMN.ReceivedAutologous, typeof(int));
			dt.Columns.Add(COLUMN.ModifiedAllogeneic, typeof(int));
			dt.Columns.Add(COLUMN.ModifiedAllogeneicPooledUnits, typeof(int));
			dt.Columns.Add(COLUMN.ModifiedDirected, typeof(int));
			dt.Columns.Add(COLUMN.ModifiedDirectedPooledUnits, typeof(int));
			dt.Columns.Add(COLUMN.ModifiedAutologous, typeof(int));
			dt.Columns.Add(COLUMN.ModifiedAutologousPooledUnits, typeof(int));
			dt.Columns.Add(COLUMN.TransfusedAllogeneic, typeof(int));
			dt.Columns.Add(COLUMN.TransfusedDirected, typeof(int));
			dt.Columns.Add(COLUMN.TransfusedAutologous, typeof(int));
			dt.Columns.Add(COLUMN.ShippedAllogeneic, typeof(int));
			dt.Columns.Add(COLUMN.ShippedDirected, typeof(int));
			dt.Columns.Add(COLUMN.ShippedAutologous, typeof(int));
			dt.Columns.Add(COLUMN.OutdatedAllogeneic, typeof(int));
			dt.Columns.Add(COLUMN.OutdatedDirected, typeof(int));
			dt.Columns.Add(COLUMN.OutdatedAutologous, typeof(int));
			dt.Columns.Add(COLUMN.DiscardedAllogeneic, typeof(int));
			dt.Columns.Add(COLUMN.DiscardedDirected, typeof(int));
			dt.Columns.Add(COLUMN.DiscardedAutologous, typeof(int));

			return dt;
		}
		

		/// <summary>
		/// GetAdministrativeDataReportRows
		/// </summary>
		/// <param name="divisionCode"></param>
		private void GetAdministrativeDataReportRows(string divisionCode)
		{
			//Red Blood Cells section
			AddOneRow(divisionCode, SECTION.RedBloodCells, SECTION.RedBloodCells);
			AddOneRow(divisionCode, SECTION.RedBloodCells, SUBDIVISION.SplitDivided);
			AddOneRow(divisionCode, SECTION.RedBloodCells, SUBDIVISION.IrradiateCells);
			AddOneRow(divisionCode, SECTION.RedBloodCells, SUBDIVISION.Leukoreduced);
			AddOneRow(divisionCode, SECTION.RedBloodCells, SUBDIVISION.WashRBC);
			AddOneRow(divisionCode, SECTION.RedBloodCells, SUBDIVISION.Deglycerolized);
			AddOneRow(divisionCode, SECTION.RedBloodCells, SUBDIVISION.CMVNegative);
			AddOneRow(divisionCode, SECTION.RedBloodCells, SUBDIVISION.SCNegative);
			AddOneRow(divisionCode, SECTION.RedBloodCells, SUBDIVISION.Frozen);
			AddOneRow(divisionCode, SECTION.RedBloodCells, SUBDIVISION.Rejuvenated);
			AddOneRow(divisionCode, SECTION.RedBloodCells, SUBDIVISION.FrozenRejuvenated);
			AddOneRow(divisionCode, SECTION.RedBloodCells, SUBDIVISION.VolumeReduce);

			//Whole Blood section
			AddOneRow(divisionCode, SECTION.WholeBlood, SECTION.WholeBlood);
			AddOneRow(divisionCode, SECTION.WholeBlood, SUBDIVISION.SplitDivided);
			AddOneRow(divisionCode, SECTION.WholeBlood, SUBDIVISION.IrradiateCells);
			AddOneRow(divisionCode, SECTION.WholeBlood, SUBDIVISION.WashRBC);
			AddOneRow(divisionCode, SECTION.WholeBlood, SUBDIVISION.Leukoreduced);
			AddOneRow(divisionCode, SECTION.WholeBlood, SUBDIVISION.CMVNegative);
			AddOneRow(divisionCode, SECTION.WholeBlood, SUBDIVISION.SCNegative);
			AddOneRow(divisionCode, SECTION.WholeBlood, SUBDIVISION.Frozen);
			AddOneRow(divisionCode, SECTION.WholeBlood, SUBDIVISION.Rejuvenated);
			AddOneRow(divisionCode, SECTION.WholeBlood, SUBDIVISION.VolumeReduce);
			
			//Fresh Frozen Plasma
			AddOneRow(divisionCode, SECTION.FreshFrozenPlasma, SECTION.FreshFrozenPlasma);
			AddOneRow(divisionCode, SECTION.FreshFrozenPlasma, SUBDIVISION.Thawed);
			AddOneRow(divisionCode, SECTION.FreshFrozenPlasma, SUBDIVISION.Leukoreduced);
			AddOneRow(divisionCode, SECTION.FreshFrozenPlasma, SUBDIVISION.SplitDivided);
			AddOneRow(divisionCode, SECTION.FreshFrozenPlasma, SUBDIVISION.Irradiated);

			//Platelets
			AddOneRow(divisionCode, SESTEOR.Vlatelets, SESTEOR.Vlatelets);
			AddOneRow(divisionCode, SESTEOR.Vlatelets, SUBDIVISION.Pooled);
			AddOneRow(divisionCode, SESTEOR.Vlatelets, SUBDIVISION.IrradiateCells);
			AddOneRow(divisionCode, SESTEOR.Vlatelets, SUBDIVISION.WashPlatelets);
			AddOneRow(divisionCode, SESTEOR.Vlatelets, SUBDIVISION.Leukoreduced);
			AddOneRow(divisionCode, SESTEOR.Vlatelets, SUBDIVISION.CMVNegative);

			//Cryoprecipitate
			AddOneRow(divisionCode, SECTION.Cryoprecipitate, SECTION.Cryoprecipitate);
			AddOneRow(divisionCode, SECTION.Cryoprecipitate, SUBDIVISION.Pooled);
			AddOneRow(divisionCode, SECTION.Cryoprecipitate, SUBDIVISION.Thawed);
			AddOneRow(divisionCode, SECTION.Cryoprecipitate, SUBDIVISION.ThawedPooled);

			//Other
			AddOneRow(divisionCode, SECTION.Other, SECTION.Other);
			AddOneRow(divisionCode, SECTION.Other, SUBDIVISION.SplitDivided);
			AddOneRow(divisionCode, SECTION.Other, SUBDIVISION.IrradiateCells);
			AddOneRow(divisionCode, SECTION.Other, SUBDIVISION.Leukoreduced);
			AddOneRow(divisionCode, SECTION.Other, SUBDIVISION.CMVNegative);
			AddOneRow(divisionCode, SECTION.Other, SUBDIVISION.Pooled);
		}


		/// <summary>
		/// AddOneRow
		/// </summary>
		/// <param name="divisionCode"></param>
		/// <param name="sectionName"></param>
		/// <param name="subdivisionName"></param>
		private void AddOneRow(string divisionCode, string sectionName, string subdivisionName)
		{
		    DataRow dr = dtReport.NewRow();
			dr[COLUMN.DivisionCode] = divisionCode;
			dr[COLUMN.ComponentClassName] = sectionName; 
			dr[COLUMN.SubdivisionName] = subdivisionName;
			dr[COLUMN.ReceivedAllogeneic] = 0;
			dr[COLUMN.ReceivedDirected] = 0; 
			dr[COLUMN.ReceivedAutologous] = 0;
			dr[COLUMN.ModifiedAllogeneic] = 0;
			dr[COLUMN.ModifiedAllogeneicPooledUnits] = 0;
			dr[COLUMN.ModifiedDirected] = 0;
			dr[COLUMN.ModifiedDirectedPooledUnits] = 0;
			dr[COLUMN.ModifiedAutologous] = 0;
			dr[COLUMN.ModifiedAutologousPooledUnits] = 0;
			dr[COLUMN.TransfusedAllogeneic] = 0;
			dr[COLUMN.TransfusedDirected] = 0;
			dr[COLUMN.TransfusedAutologous] = 0;
			dr[COLUMN.ShippedAllogeneic] = 0;
			dr[COLUMN.ShippedAllogeneic] = 0;
			dr[COLUMN.ShippedDirected] = 0;
			dr[COLUMN.ShippedAutologous] = 0;
			dr[COLUMN.OutdatedAllogeneic] = 0;
			dr[COLUMN.OutdatedDirected] = 0;
			dr[COLUMN.OutdatedAutologous] = 0;
			dr[COLUMN.DiscardedAllogeneic] = 0;
			dr[COLUMN.DiscardedDirected] = 0;
			dr[COLUMN.DiscardedAutologous] = 0;

			dtReport.Rows.Add(dr);
		}


		/// <summary>
		/// CheckDivision
		/// </summary>
		/// <param name="dr"></param>
		private void CheckDivision(DataRow dr)
		{
			string filterExpression = COLUMN.DivisionCode + " = '" +
				dr[TABLE.BloodUnit.DivisionCode].ToString() + "'";

			dtExisting = dtReport.Select(filterExpression);
			
			//If there are no rows for passed division, creat new
			if(dtExisting.Length == 0)
			{
				this.GetAdministrativeDataReportRows(dr[TABLE.BloodUnit.DivisionCode].ToString());
				//filter again so we get empty rows
				dtExisting = dtReport.Select(filterExpression);
			}
		}

		/// <summary>
		/// UpdateTotals
		/// </summary>
		/// <param name="rowNumber"></param>
		/// <param name="donationTypeID"></param>
		/// <param name="allogeneicColumn"></param>
		/// <param name="directedColumn"></param>
		/// <param name="autologousColumn"></param>
		private void UpdateTotals(int rowNumber, int donationTypeID, 
								string allogeneicColumn, string directedColumn, string autologousColumn)
		{
			if(IsAllogeneic(donationTypeID))
			{
				dtExisting[rowNumber][allogeneicColumn] = (int) dtExisting[rowNumber][allogeneicColumn] + 1;  
			}
			else if(Common.Utility.IsAutologous(donationTypeID))
			{
				dtExisting[rowNumber][autologousColumn] = (int) dtExisting[rowNumber][autologousColumn] + 1;  
			}
			else if(Common.Utility.IsDirected(donationTypeID))
			{
				dtExisting[rowNumber][directedColumn] = (int) dtExisting[rowNumber][directedColumn] + 1;  
			}
		}

		/// <summary>
		/// IsAllogeneic
		/// </summary>
		/// <param name="donationTypeID"></param>
		/// <returns></returns>
		private bool IsAllogeneic(int donationTypeID)
		{
			//allogeneic units -> donationTypeID = 2
			if(donationTypeID == 2)
			{
				return true;
			}
			else
			{
				return false;
			}
		}

		/// <summary>
		/// IsCodabarSplit
		/// </summary>
		/// <param name="dr"></param>
		/// <returns></returns>
		private bool IsCodabarSplit(DataRow dr)
		{			
			//Codabar Label Type Code
			if(dr[TABLE.BloodUnit.LabelTypeCode].ToString().ToUpper() == "C")
			{
				Regex splitEyeReadableUnitID;

			    splitEyeReadableUnitID = Common.RegularExpressions.EyeReadableSuffixAlpha();
				string lastChar = dr[TABLE.BloodUnit.EyeReadableUnitId].ToString().Substring(dr[TABLE.BloodUnit.EyeReadableUnitId].ToString().Length - 1).ToUpper();

				return (splitEyeReadableUnitID.IsMatch(lastChar));
			}
			else
			{
				return false;
			}
		}

		/// <summary>
		/// IsISBTSplit
		/// </summary>
		/// <param name="dr"></param>
		/// <returns></returns>
		private bool IsISBTSplit(DataRow dr)
		{
			//ISBT Label Type Code
			if(dr[TABLE.BloodUnit.LabelTypeCode].ToString().ToUpper() == "I")
			{
				if(dr[TABLE.BloodUnit.UnitDivisionId] == DBNull.Value ||
					dr[TABLE.BloodUnit.UnitDivisionId].ToString().Length == 0)
				{
					return false;
				}
				else if((dr[TABLE.BloodUnit.UnitDivisionId].ToString().Substring(1,1) == "0"))
				{
					return false;					
				}
				else
				{
					//Any number other than 0 in 2nd char of unit division id
					return true;
				}
			}
			else
			{
				return false;
			}
		}

		#endregion


		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>4/26/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="7260"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="7261"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Generate and print report
		/// Used by VbecsConsoleReport application
		/// </summary>
        protected override VbecsReportPrinter createReportPrinter()
		{
            BOL.VbecsUser user = new BOL.VbecsUser(this.PrintedBy);
            DataTable dtReportData = this.GetAdministrativeDataReport();

            string reportCriteria = "Date Range: ";

            reportCriteria += Common.VBECSDateTime.FormatDateString(this.StartDate);
            reportCriteria += " - ";
            reportCriteria += Common.VBECSDateTime.FormatDateString(this.EndDate);

            if (this.IsMultiDivisional)
            {
                reportCriteria += " \nMulti-divisional";
            }
            else
            {
                reportCriteria += " \nDivision: ";
                reportCriteria += Common.LogonUser.LogonUserDivisionCode;
            }


            DataSet dsHeader = BOL.VbecsReport.GetReportHeader(user.UserName);
            DataSet dsFooter = BOL.VbecsReport.GetReportFooter(reportCriteria);

            ReportDataSource reportSource = new ReportDataSource();
            if (dtReportData != null)
            {
                reportSource.Name = "AdministrativeReportDataSet";
                reportSource.Value = dtReportData;
            }

            ReportDataSource reportSourceHeader = new ReportDataSource();
            if (dsHeader.Tables.Count > 0)
            {
                reportSourceHeader.Name = "ReportHeaderDataSet";
                reportSourceHeader.Value = dsHeader.Tables[0];
            }
            ReportDataSource reportSourceFooter = new ReportDataSource();
            if (dsHeader.Tables.Count > 0)
            {
                reportSourceFooter.Name = "ReportFooterDataSet";
                reportSourceFooter.Value = dsFooter.Tables[0];
            }

            Assembly assembly = Assembly.LoadFrom(Path.GetDirectoryName(Assembly.GetExecutingAssembly().Location) + "\\Reports.dll");
            Stream stream = assembly.GetManifestResourceStream("gov.va.med.vbecs.reports.AdministrativeData.AdministrativeDataReport.rdlc");
            LocalReport report = new LocalReport();
            report.DataSources.Add(reportSource);
            report.DataSources.Add(reportSourceHeader);
            report.DataSources.Add(reportSourceFooter);
            report.LoadReportDefinition(stream);
            return new BOL.VbecsReportPrinter(this.PrinterName, true, report);
		}
	}
}
